const communityService = require("./club.service");
const { success, failure, unauthorized, serverError } = require("../../../rider_connect_shared_repo/src/utils/responseHandler");

async function createClubController(req, res) {
    try {
        const {
            club_mode,
            club_logo_url,
            club_cover_image_url,
            club_name,
            club_motto,
            description,
            club_vision,
            club_goal,
            location,
            language,
            rules_regulations,
            driving_license,
            club_privacy_type,
            gender,
            gender_restriction,
            limit_registration,
            club_joining_fee,
            is_approved,
            admin,
            is_active
        } = req.body
        const data = {
            club_mode,
            club_logo_url,
            club_cover_image_url,
            club_name,
            club_motto,
            description,
            club_vision,
            club_goal,
            location,
            language,
            rules_regulations,
            driving_license,
            club_privacy_type,
            gender,
            gender_restriction,
            limit_registration,
            club_joining_fee,
            is_approved,
            admin,
            is_active
        };
        for (let key in data) {
            if (data[key] === undefined || data[key] === null || (typeof data[key] === "string" && data[key].trim() === "")) {
                return res.status(400).json({ message: `${key} is required.` });
            }
        }

        const result = await communityService.createClubService(data);
        if (!result) return failure(res, "Failed");
        success(res, result, "Club Created Successfully");
    } catch (error) {
        serverError(res, error.message);
    }
}

async function updateClubController(req, res) {
    try {
        const id = req.params.id;
        if (!id) return res.status(400).json({ message: "Club ID is required." });

        const {
            club_mode,
            club_logo_url,
            club_cover_image_url,
            club_name,
            club_motto,
            description,
            club_vision,
            club_goal,
            location,
            language,
            rules_regulations,
            driving_license,
            club_privacy_type,
            gender,
            gender_restriction,
            limit_registration,
            club_joining_fee,
            is_approved,
            admin,
            is_active
        } = req.body;

        const data = {
            club_mode,
            club_logo_url,
            club_cover_image_url,
            club_name,
            club_motto,
            description,
            club_vision,
            club_goal,
            location,
            language,
            rules_regulations,
            driving_license,
            club_privacy_type,
            gender,
            gender_restriction,
            limit_registration,
            club_joining_fee,
            is_approved,
            admin,
            is_active
        };

        for (let key in data) {
            if (data[key] === undefined || data[key] === null || (typeof data[key] === "string" && data[key].trim() === "")) {
                return res.status(400).json({ message: `${key} is required.` });
            }
        }

        const result = await communityService.updateClubService(id, data);
        if (!result) return failure(res, "Failed");
        return success(res, result, "Club Updated Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function getClubByIdController(req, res) {
    try {
        const { id } = req.params;
        if (!id) return res.status(400).json({ message: "Club ID is required." });

        const club = await communityService.getClubByIdService(id);

        if (!club) {
            return res.status(404).json({
                status: "fail",
                message: `Club with ID ${id} not found`,
                data: null
            });
        }

        return success(res, club, "Club Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function getMyClubsController(req, res) {
    try {
        const user_id = req.params.id;
        if (!user_id) return res.status(400).json({ message: "User ID is required." });

        const clubs = await communityService.getMyClubsService(user_id);

        if (!clubs || clubs.length === 0) {
            return success(res, {}, "Failed");
        }

        return success(res, clubs, "Clubs Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function getClubMembersController(req, res) {
    try {
        const club_id = req.params.id;

        if (!club_id) {
            return res.status(400).json({ message: "Club ID is required." });
        }

        const members = await communityService.getClubMembersService(club_id);

        if (!members || members.length === 0) {
            return success(res, {}, "Failed");
        }

        return success(res, members, "Members Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function createClubPostController(req, res) {
    try {
        const {
            user_id,
            club_id,
            caption,
            distance,
            duration,
            post_url,
            is_active = false
        } = req.body;

        const requiredFields = { user_id, club_id, caption, post_url };
        for (let key in requiredFields) {
            if (
                requiredFields[key] === undefined ||
                requiredFields[key] === null ||
                (typeof requiredFields[key] === "string" && requiredFields[key].trim() === "")
            ) {
                return failure(res, {}, { message: `${key} is required` });

            }
        }

        const data = {
            user_id,
            club_id,
            caption,
            distance,
            duration,
            post_url,
            is_active
        };

        const post = await communityService.createClubPostService(data);

        if (!post) return failure(res, "Failed to create post");

        return success(res, post, "Club post created successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function getClubPostController(req, res) {
    try {
        const club_id = req.params.id;

        if (!club_id) {

            return failure(res, "Club ID is required");

        }

        const posts = await communityService.getClubPostService(club_id);

        if (!posts || posts.length === 0) {
            return failure(res, "No posts found for this club");
        }

        return success(res, posts, "Club posts fetched successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function createLeaderBoardController(req, res) {
    try {
        const { club_id, leading_distance, total_club_distance } = req.body;

        const requiredFields = { club_id, leading_distance, total_club_distance };
        for (let key in requiredFields) {
            if (
                requiredFields[key] === undefined ||
                requiredFields[key] === null ||
                (typeof requiredFields[key] === "string" && requiredFields[key].trim() === "")
            ) {
                return failure(res, `${key} is required`);
            }
        }

        const data = { club_id, leading_distance, total_club_distance };


        const leaderboard = await communityService.createLeaderBoardService(data);

        if (!leaderboard) return failure(res, "Failed to create leaderboard entry");

        return success(res, leaderboard, "Leaderboard entry created successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function joinClubController(req, res) {
    try {
        const {
            club_id,
            user_id,
            role,
            joined_at,
            is_approved = false,
            is_active = false
        } = req.body;
        const requiredFields = { user_id, role, joined_at, club_id };
        const missingFields = [];

        for (const [key, value] of Object.entries(requiredFields)) {
            if (
                value === undefined ||
                value === null ||
                (typeof value === "string" && value.trim() === "")
            ) {
                missingFields.push(key);
            }
        }

        if (missingFields.length > 0) {
            return res.status(400).json({
                status: "fail",
                message: `Missing required fields: ${missingFields.join(", ")}`,
                data: null
            });
        }

        if (typeof user_id !== "number") {
            return res.status(400).json({ status: "fail", message: "user_id must be a number" });
        }

        if (typeof role !== "string") {
            return res.status(400).json({ status: "fail", message: "role must be a string" });
        }

        const data = {
            club_id,
            user_id,
            role,
            joined_at,
            is_approved,
            is_active
        };

        const result = await communityService.joinClubService(data);

        return success(res, result, "Joined group successfully", 201);
    } catch (error) {
        serverError(res, error.message);
    }
}


async function getLeaderBoardController(req, res) {
    try {
        const { id: club_id } = req.params;
        if (!club_id) {
            return failure(res, "Club ID is required");
        }

        const leaderboard = await communityService.getLeaderBoardService(club_id);

        if (!leaderboard || leaderboard.length === 0) {
            return failure(res, "No leaderboard found for this club");
        }
        return success(res, leaderboard, "Leaderboard fetched successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function exploreClubController(req, res) {
    try {
        const user_id = req.params.id;
        if (!user_id) {
            return failure(res, "User ID is required");
        }

        const clubs = await communityService.exploreClubService(user_id);

        if (!clubs || clubs.length === 0) {
            return success(res, [], "No clubs available for this user");
        }

        return success(res, clubs, "Clubs fetched successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}

async function requestedClubsController(req, res) {
    try {
        const user_id = req.params.id;

        if (!user_id) {
            return failure(res, "User ID is required");
        }

        const requested_clubs = await communityService.requestedClubsService(user_id);

        if (!requested_clubs || requested_clubs.length === 0) {
            return success(res, [], "No requested clubs found for this user");
        }

        return success(res, requested_clubs, "Requested clubs fetched successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}



async function getClubEventsController(req, res) {
    try {
        const club_id = req.params.id;

        if (!club_id) {
            return failure(res, "Club ID is required");
        }

        const club_events = await communityService.getClubEventsService(club_id);
        if (!club_events || Object.keys(club_events).length === 0) {
            return failure(res, "No event found with the provided ID");
        }

        return success(res, club_events, "Club Events Fetched Successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}

async function clubPostLikeController(req, res) {
    try {
        const { post_id, user_id } = req.body;
        const requiredFields = { post_id, user_id };
        for (let key in requiredFields) {
            if (
                requiredFields[key] === undefined ||
                requiredFields[key] === null ||
                (typeof requiredFields[key] === "string" && requiredFields[key].trim() === "")
            ) {
                return failure(res, {}, { message: `${key} is required` });
            }
        }

        const data = { post_id, user_id };

        const like = await communityService.clubPostLikeService(data);

        if (!like) {
            return failure(res, {}, { message: "Failed to like post" });
        }

        return success(res, like, "Post liked successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function clubPostUnlikeController(req, res) {
    try {
        const like_id = req.params.id;
        if (!like_id) {
            return failure(res, {}, { message: "like_id is required" });
        }

        const unlike = await communityService.clubPostUnlikeService(like_id);

        if (!unlike) {
            return failure(res, {}, { message: "Failed to unlike post" });
        }

        return success(res, {}, "Post unliked successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function commentClubPostController(req, res) {
    try {
        const {
            club_id,
            post_id,
            user_id,
            is_reply = false,
            id_reply_for = null,
            comment
        } = req.body;


        const requiredFields = { club_id, post_id, user_id, comment };
        for (let key in requiredFields) {
            if (
                requiredFields[key] === undefined ||
                requiredFields[key] === null ||
                (typeof requiredFields[key] === "string" && requiredFields[key].trim() === "")
            ) {
                return failure(res, {}, { message: `${key} is required` });
            }
        }

        if (is_reply && !id_reply_for) {
            return failure(res, {}, { message: "id_reply_for is required when is_reply is true" });
        }

        const data = { club_id, post_id, user_id, is_reply, id_reply_for, comment };

        const newComment = await communityService.commentClubPostService(data);

        if (!newComment) {
            return failure(res, {}, { message: "Failed to add comment" });
        }

        return success(res, newComment, "Comment added successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function deleteCommentController(req, res) {
    try {
        const id = req.params.id;


        if (!id) {
            return failure(res, {}, { message: "Comment ID is required" });
        }

        const deleted = await communityService.deleteCommentService(id);

        if (!deleted) {
            return failure(res, {}, { message: "Failed to delete comment or comment not found" });
        }

        return success(res, {}, "Comment deleted successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}



module.exports = {
    createClubController,
    updateClubController,
    getClubByIdController,
    getMyClubsController,
    getClubMembersController,
    createClubPostController,
    getClubPostController,
    createLeaderBoardController,
    joinClubController,
    getLeaderBoardController,
    exploreClubController,
    requestedClubsController,
    getClubEventsController,
    clubPostLikeController,
    clubPostUnlikeController,
    commentClubPostController,
    deleteCommentController

}
