const communityService = require("./event.service");
const { success, failure, unauthorized, serverError } = require("../../../rider_connect_shared_repo/src/utils/responseHandler");


async function createEventRideController(req, res) {
    try {
        const {
            event_image_url,
            event_name,
            description,
            location,
            event_type,
            start_date,
            end_date,
            start_time,
            end_time,
            riding_skill,
            is_limit,
            bike_cc,
            mentioned_club,
            mentioned_group,
            reg_start_date,
            reg_fee,
            rsvp,
            contact_person,
            contact,
            is_active,
            is_approved,
            event_code,
            user_id
        } = req.body;

        const data = {
            event_image_url,
            event_name,
            description,
            location,
            event_type,
            start_date,
            end_date,
            start_time,
            end_time,
            riding_skill,
            is_limit,
            bike_cc,
            mentioned_club,
            mentioned_group,
            reg_start_date,
            reg_fee,
            rsvp,
            contact_person,
            contact,
            is_active,
            is_approved,
            event_code,
            user_id
        };


        const event = await communityService.createEventRideService(data);

        return success(res, event, 'Event ride created successfully');


    } catch (error) {
        return serverError(res, error.message);
    }
}


async function getEventByIdController(req, res) {
    try {
        const event_id = req.params.id;

        if (!event_id) {
            return failure(res, "Event ID is required");
        }

        const event = await communityService.getEventByIdService(event_id);

        if (!event || Object.keys(event).length === 0) {
            return failure(res, "No event found with the provided ID");
        }

        return success(res, event, "Event Fetched Successfully");
    } catch (error) {
        return serverError(res, error.message);
    }
}


async function joinEventController(req, res) {
    try {
        const {
            event_id,
            user_id,
            role,
            joined_at,
            is_approved,
            is_active
        } = req.body;

        if (
            !event_id ||
            !user_id ||
            !role ||
            !joined_at ||
            (is_approved === undefined || is_approved === null) ||
            (is_active === undefined || is_active === null)
        ) {
            return failure(res, "All fields are mandatory");
        }

        const data = {
            event_id,
            user_id,
            role,
            joined_at,
            is_approved,
            is_active
        };

        const join_event = await communityService.joinEventService(data);

        if (!join_event) {
            return failure(res, "Failed to join event, please try again");
        }

        return success(res, join_event, "Joined event successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}


async function getEventMembersController(req, res) {
    try {
        const event_id = req.params.id;

        if (!event_id) {
            return res.status(400).json({ message: "Event ID is required." });
        }

        const members = await communityService.getEventMembersService(event_id);

        if (!members || members.length === 0) {
            return success(res, {}, "Failed");
        }

        return success(res, members, "Members Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function createEventPostController(req, res) {
    try {
        const {
            user_id,
            event_id,
            caption,
            post_url,
            is_active = true
        } = req.body;

        const requiredFields = { user_id, event_id, caption, post_url };
        for (let key in requiredFields) {
            if (
                requiredFields[key] === undefined ||
                requiredFields[key] === null ||
                (typeof requiredFields[key] === "string" && requiredFields[key].trim() === "")
            ) {
                return failure(res, {}, { message: `${key} is required` });
            }
        }

        const data = { user_id, event_id, caption, post_url, is_active };

        const post = await communityService.createEventPostService(data);

        if (!post) return failure(res, {}, { message: "Failed to create post" });

        return success(res, post, "Event post created successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}


async function getAllEventPostController(req, res) {
    try {
        const event_id = req.params.id;

        if (!event_id) {

            return failure(res, "Event ID is required");

        }

        const posts = await communityService.getAllEventPostService(event_id);

        if (!posts || posts.length === 0) {
            return failure(res, "No posts found for this Event");
        }

        return success(res, posts, "Event posts fetched successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}




module.exports = {
    createEventRideController,
    getEventByIdController,
    joinEventController,
    getEventMembersController,
    createEventPostController,
    getAllEventPostController

}