const communityService = require("./group.service");
const { success, failure, unauthorized, serverError } = require("../../../rider_connect_shared_repo/src/utils/responseHandler");



async function createGroupController(req, res) {
    try {
        const {
            cover_image_url,
            group_name,
            description,
            privacy_type,
            location,
            is_approved,
            is_active,
            user_id
        } = req.body;

        if (
            !cover_image_url ||
            !group_name ||
            !description ||
            !privacy_type ||
            !location ||
            (is_approved === undefined || is_approved === null) ||
            (is_active === undefined || is_active === null) ||
            !user_id
        ) {
            return failure(res, "All fields are mandatory");
        }

        const data = {
            cover_image_url,
            group_name,
            description,
            privacy_type,
            location,
            is_approved,
            is_active,
            user_id
        };

        const group = await communityService.createGroupService(data);
        if (!group) {
            return failure(res, "Failed to create group, please try again");
        }
        return success(res, group, "Group created successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}

async function joinGroupController(req, res) {
    try {
        const {
            group_id,
            user_id,
            joined_at,
            role,
            is_approved,
            is_active
        } = req.body;

        if (
            !group_id ||
            !user_id ||
            !joined_at ||
            !role ||
            (is_approved === undefined || is_approved === null) ||
            (is_active === undefined || is_active === null)
        ) {
            return failure(res, "All fields are mandatory");
        }

        const data = {
            group_id,
            user_id,
            joined_at,
            role,
            is_approved,
            is_active
        };

        const join_group = await communityService.joinGroupService(data);

        if (!join_group) {
            return failure(res, "Failed to join group, please try again");
        }

        return success(res, join_group, "Joined group successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}


async function requestedGroupsController(req, res) {
    try {
        const user_id = req.params.id;

        if (!user_id) {
            return failure(res, "User ID is required");
        }

        const requested_groups = await communityService.requestedGroupsService(user_id);

        if (!requested_groups || requested_groups.length === 0) {
            return success(res, [], "No requested groups found for this user");
        }

        return success(res, requested_groups, "Requested groups fetched successfully");
    } catch (error) {

    }
}

async function exploreGroupsController(req, res) {
    try {
        const user_id = req.params.id;
        if (!user_id) {
            return failure(res, "User ID is required");
        }

        const groups = await communityService.exploreGroupsService(user_id);

        if (!groups || groups.length === 0) {
            return success(res, [], "No groups available for this user");
        }

        return success(res, groups, "Groups fetched successfully");

    } catch (error) {
        return serverError(res, error.message);
    }
}


async function getMyGroupsController(req, res) {
    try {
        const user_id = req.params.id;
        if (!user_id) return res.status(400).json({ message: "User ID is required." });

        const groups = await communityService.getMyGroupsService(user_id);

        if (!groups || groups.length === 0) {
            return success(res, {}, "Failed");
        }

        return success(res, groups, "Groups Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function getGroupMembersController(req, res) {
    try {
        const group_id = req.params.id;

        if (!group_id) {
            return res.status(400).json({ message: "Group ID is required." });
        }

        const members = await communityService.getGroupMembersService(group_id);

        if (!members || members.length === 0) {
            return success(res, {}, "Failed");
        }

        return success(res, members, "Members Fetched Successfully");

    } catch (error) {
        serverError(res, error.message);
    }
}

async function getGroupEventsController(req, res) {
    try {
        const group_id = req.params.id;

        if (!group_id) {
            return failure(res, "Group ID is required");
        }

        const group_events = await communityService.getGroupEventsService(group_id);
        if (!group_events || Object.keys(group_events).length === 0) {
            return failure(res, "No event found with the provided ID");
        }

        return success(res, group_events, "Group Events Fetched Successfully");
    } catch (error) {
        serverError(res, error.message);
    }
}




module.exports = {
    createGroupController,
    joinGroupController,
    requestedGroupsController,
    exploreGroupsController,
    getMyGroupsController,
    getGroupMembersController,
    getGroupEventsController
}