// index.js (or startup.js)
const { spawn } = require('child_process');
const path = require('path');

// Define the services, their file paths, and their *working directories*
const services = [
    // NOTE: Changed 'gateway' to 'api-gateway' based on your provided code structure
    { name: 'API Gateway', file: 'server.js', cwd: 'api-gateway' }, 
    { name: 'Auth Service', file: 'index.js', cwd: 'auth-service' },
    { name: 'Community Service', file: 'index.js', cwd: 'community-service' },
    { name: 'Ride Craft Service', file: 'index.js', cwd: 'ride-craft-service' }
];

console.log('🚀 Starting all microservices...');

services.forEach(service => {
    // Determine the absolute path for the working directory
    const serviceCwd = path.join(process.cwd(), service.cwd);

    // Spawn a new node process for each service
    const child = spawn('node', [service.file], {
        stdio: 'inherit',
        shell: true,
        cwd: serviceCwd,
        // **CRITICAL FIX:** Detach the child process from the parent's event loop
        detached: true
    });

    // **MISSING CRITICAL FIX:** Allow the parent script to exit immediately
    // without waiting for the child processes to complete.
    child.unref(); 

    // The 'exit' handler is now mainly for quick failures, as the parent
    // will exit before the children normally terminate.
    child.on('error', (err) => {
        console.error(`❌ ERROR launching ${service.name} (${service.file}):`, err);
    });

    child.on('exit', (code, signal) => {
        if (code !== 0 && code !== null) {
             console.error(`🛑 ${service.name} process failed with code ${code} and signal ${signal}`);
        } else {
             // This log will appear only if a service exits almost instantly
             console.warn(`🛑 ${service.name} process exited with code ${code} and signal ${signal}`);
        }
    });

    console.log(`✅ Successfully started ${service.name}. PID: ${child.pid}`);
});

console.log('\n---');
// Updated message to reflect the new detached behavior
console.log('All services launched in the background. Parent script is now exiting.');
// The script should exit immediately here.