const model = require("../../models");
const { Op } = require("sequelize");

const User = model.User;
const user_vehicle_details = model.user_vehicle_details;


async function findUserByEmailPhoneId(data) {
    const { id, email, phone } = data;

    if (!id && !email && !phone) {
        throw new Error("At least one of ID, Email, or Mobile is required");
    }

    const whereClause = { [Op.or]: [] };
    if (id) whereClause[Op.or].push({ id });
    if (email) whereClause[Op.or].push({ email });
    if (phone) whereClause[Op.or].push({ phone });

    const user = await User.findOne({ where: whereClause });

    // Return null instead of throwing
    return user || null;
}

/**
 * Find a user by email or phone and password
 */
async function findUserByEmailOrMobileAndPassword(data) {
    const { email = null, phone = null, password } = data;

    if (!password) throw new Error("Password is required");
    if (!email && !phone) throw new Error("Either email or mobile must be provided");

    const whereClause = {
        password,
        [Op.or]: []
    };

    if (email) whereClause[Op.or].push({ email });
    if (phone) whereClause[Op.or].push({ phone });

    return await User.findOne({ where: whereClause });
}

/**
 * Create a new user
 */
async function createUser(data) {
    const { email, phone, password, referral_number } = data;
    return await User.create({ email, phone, password, referral_number });
}

/**
 * Update OTP for a user
 */
async function updateOtp(data) {
    const { phone = null, email = null, otp } = data;

    const whereClause = {};
    if (phone) whereClause.phone = phone;
    else if (email) whereClause.email = email;
    else throw new Error("Either mobile or email must be provided");

    return await User.update({ otp }, { where: whereClause });
}

/**
 * Verify OTP
 */
async function otpVerify(data) {
    const { id, otp } = data;
    if (!id) throw new Error("User ID is required");
    if (!otp) throw new Error("OTP is required");

    return await User.findOne({ where: { id, otp } });
}

/**
 * Setup / Update User Profile
 */
async function setupProfile(data) {
    const { id, first_name, last_name, user_name, gender, dob } = data;
    if (!id) throw new Error("User ID is required");

    const updateData = {};
    if (first_name !== undefined) updateData.first_name = first_name;
    if (last_name !== undefined) updateData.last_name = last_name;
    if (user_name !== undefined) updateData.user_name = user_name;
    if (gender !== undefined) updateData.gender = gender;
    if (dob !== undefined) updateData.dob = dob;

    const [updatedRows] = await User.update(updateData, { where: { id } });

    if (updatedRows === 0) throw new Error("User not found or no fields to update");

    return await User.findOne({ where: { id } });
}

/**
 * Set Vehicle Details
 */
async function setVehicleDetail(data) {
    if (data.is_active === undefined) data.is_active = true;
    return await user_vehicle_details.create(data);
}

async function findUserVehicle(data) {
    const { user_id } = data;

    const vehicle = await user_vehicle_details.findAll({ where:{ user_id: user_id }});

    
    return vehicle || null;
}

async function updatePassword(data) {
    const { id, password } = data;

    if (!id) throw new Error("User ID is required");
    if (!password) throw new Error("Password is required");

    const [updatedRows] = await User.update({ password }, { where: { id } });

    if (updatedRows === 0) throw new Error("User not found or password not updated");

    return await User.findOne({ where: { id } });
}

module.exports = {
    findUserByEmailPhoneId,
    findUserByEmailOrMobileAndPassword,
    createUser,
    updateOtp,
    otpVerify,
    setupProfile,
    setVehicleDetail,
    updatePassword,
    findUserVehicle
};
