const model = require("../../models");
const { Op, where } = require("sequelize");

const user = model.User;
const club = model.club_data;
const club_members = model.club_members;
const club_post = model.club_post;
const leaderboard_master = model.leaderboard_master;
const leaderboard_members = model.leaderboard_members;
const events_rides_master = model.events_rides_master;
const club_post_likes=model.club_post_likes;
const club_post_comments=model.club_post_comments;

async function createClubFunction(data) {
    return await club.create(data);
}

async function updateClubFunction(id, data) {
    return await club.update(data, { where: { id }, returning: true });
}

async function getClubByIdFunction(id) {
    return await club.findByPk(id);
}

async function getMyClubsFunction(user_id) {

    const memberships = await club_members.findAll({
        where: { user_id },
        attributes: ["club_id"]
    });

    if (!memberships || memberships.length === 0) return [];

    const clubIds = memberships.map(m => m.club_id);

    const clubs = await club.findAll({
        where: {
            id: { [Op.in]: clubIds }
        }
    });

    return clubs;
}


async function getClubMembersFunction(club_id) {

    const memberships = await club_members.findAll({
        where: { club_id },
        attributes: ["user_id", "role", "club_id"]
    });

    if (!memberships || memberships.length === 0) return [];

    const user_ids = memberships.map(m => m.user_id);

    const users = await user.findAll({
        where: { id: { [Op.in]: user_ids } },
        attributes: ["id", "first_name", "last_name", "email", "phone"]
    });

    const membersWithDetails = memberships.map(m => {
        const u = users.find(user => user.id === m.user_id);
        return {
            user_id: u.id,
            first_name: u.first_name,
            last_name: u.last_name,
            email: u.email,
            mobile: u.phone,
            role: m.role,
            club_id: m.club_id
        };
    });

    return membersWithDetails;
}

async function createClubPostFunction(data) {
    return await club_post.create(data);
}

async function getClubPostFunction(club_id) {
    const post = await club_post.findAll({ where: { club_id } });
    return post;
}

async function joinClubFunction(data) {
    return await club_members.create(data);
}

async function createLeaderBoardFunction(data) {
    return await leaderboard_master.create(data);
}

async function getClubLeaderBoardFunction(club_id) {
    const leaderboard = await leaderboard_master.findOne({ where: { club_id } });

    if (!leaderboard) return [];

    const { id: leaderboard_id, leading_distance, total_club_distance } = leaderboard;
    const members = await leaderboard_members.findAll({
        where: { leaderboard_id },
        attributes: ["user_id", "distance"]
    });

    if (!members || members.length === 0) return [];

    const user_ids = members.map(m => m.user_id);

    const users = await user.findAll({
        where: { id: { [Op.in]: user_ids } },
        attributes: ["id", "first_name", "last_name", "user_name", "email"]
    });

    const leaderboardDetails = members.map(m => {
        const u = users.find(user => user.id === m.user_id);
        return {
            user_id: u.id,
            first_name: u.first_name,
            last_name: u.last_name,
            user_name: u.user_name,
            email: u.email,
            distance: m.distance,
            leaderboard_id,
            leading_distance,
            total_club_distance
        };
    });

    return leaderboardDetails;
}


async function exploreClubFunction(user_id) {
    try {
        const userClubs = await club_members.findAll({
            where: { user_id },
            attributes: ["club_id"],
            raw: true
        });

        const joinedClubIds = userClubs.map(c => c.club_id);

        const clubs = await club.findAll({
            where: {
                id: {
                    [Op.notIn]: joinedClubIds.length > 0 ? joinedClubIds : [0]
                }
            }
        });

        return clubs;
    } catch (error) {
        throw error;
    }
}

async function requestedClubsFunction(user_id) {
    try {
        const requestedClubs = await club_members.findAll({
            where: {
                user_id,
                is_approved: false
            },
            attributes: ["club_id"],
            raw: true
        });

        const clubIds = requestedClubs.map(c => c.club_id);

        if (clubIds.length === 0) {
            return [];
        }

        const clubs = await club.findAll({
            where: {
                id: {
                    [Op.in]: clubIds
                }
            }
        });

        return clubs;
    } catch (error) {

        throw error;
    }
}

async function getClubEventsFunction(club_id) {
    const event = await events_rides_master.findAll({ where: { mentioned_club: club_id } });
    return event;
}

async function clubPostLikeFunction(data){
    const like = await club_post_likes.create(data);
    return like;
}

async function clubPostUnlikeFunction(id) {
    const like = await club_post_likes.destroy({
        where: { id }
    });
    return like; 
}

async function commentClubPostFunction(data){
    const comment=await club_post_comments.create(data);
    return comment;
}

async function deleteCommentFunction(id){
    const comment = await club_post_comments.destroy({where:{id}});
    return comment
}

module.exports = {
    createClubFunction,
    updateClubFunction,
    getClubByIdFunction,
    getMyClubsFunction,
    getClubMembersFunction,
    createClubPostFunction,
    getClubPostFunction,
    createLeaderBoardFunction,
    joinClubFunction,
    getClubLeaderBoardFunction,
    exploreClubFunction,
    requestedClubsFunction,
    getClubEventsFunction,
    clubPostLikeFunction,
    clubPostUnlikeFunction,
    commentClubPostFunction,
    deleteCommentFunction
}